/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   System Reboot Service
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file RebootControl.h
 \brief System reboot into hardware bootstrap API

*/

#ifndef __REBOOTCONTROL_H
#define __REBOOTCONTROL_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <NativeError.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid boot
           control attributes in the S_RebootControlAttributes structures. */
typedef enum
{
    USE_INITIAL_PROTOCOL = 0x0001,              /*!< Selects S_RebootControlAttributes::bUseInitialProtocol */
    USE_BOOT_PROTOCOL = 0x0002,                 /*!< Selects S_RebootControlAttributes::BootProtocol */
    USE_BOOT_ACCESS_LEVEL = 0x0004,             /*!< Selects S_RebootControlAttributes::BootAccessLevel */
    USE_BOOT_CITY_ID = 0x0008,                  /*!< Selects S_RebootControlAttributes::BootCityID */
    USE_BOOT_DEFAULT_KEY_SERIAL_BAUD = 0x0010,  /*!< Selects S_RebootControlAttributes::BootDefaultKeySerialBaudRate */
    USE_BOOT_CURR_SERIAL_BAUD = 0x0020,         /*!< Selects S_RebootControlAttributes::BootCurrentSerialBaudRate */
    USE_BOOT_CURR_CAN_BITRATE = 0x0040,         /*!< Selects S_RebootControlAttributes::BootCANBitRate */
    USE_REBOOT_APPLICATION = 0x0080,            /*!< Will reboot the main application via the hardboot */

/* IF THIS TYPE EXCEEDS 0x8000 THEN ALTER THE TYPE OF THE RebootControlValidAttributesMask_U ACCORDINGLY */

} E_RebootControlValidAttributesMask;

/*! Unsigned integer type of sufficient size to hold the attribute masks for a the reboot control described by \ref E_RebootControlValidAttributesMask */
typedef uint2 RebootControlValidAttributesMask_U;

/*! \brief A RebootControl buffer contains a series of control attributes which will be passed to hardware bootstrap application.

 Use S_RebootControlAttributes::uValidAttributesMask to define which attributes are valid. */
typedef struct
{
/*! Describes which attributes in \ref S_RebootControlAttributes are to be applied. */
    RebootControlValidAttributesMask_U uValidAttributesMask;    

/*! Indicates whether the hardware bootstrap code should use begin communications based on the following parameters. Select with \ref USE_INITIAL_PROTOCOL */
    bool1   bUseInitialProtocol;
/*! When \ref bUseInitialProtocol is set, instructs the hardware bootstrap application to startup using this protocol type. Select with \ref USE_BOOT_PROTOCOL */
    uint1   u1BootProtocol;
/*! When \ref bUseInitialProtocol is set, instructs the hardware bootstrap application to startup using this access level type. Select with \ref USE_BOOT_ACCESS_LEVEL */
    uint1   u1BootAccessLevel;
/*! When \ref bUseInitialProtocol is set, instructs the hardware bootstrap application to startup using this city id type. Select with \ref USE_BOOT_CITY_ID */
    uint1   u1BootCityID;
/*! When \ref bUseInitialProtocol is set, instructs the hardware bootstrap application KEY protocol to use this default baud rate. Select with \ref USE_BOOT_DEFAULT_KEY_SERIAL_BAUD */
    uint4   u4BootDefaultKeySerialBaudRate;
/*! When \ref bUseInitialProtocol is set, instructs the hardware bootstrap application to startup using this baud rate. Select with \ref USE_BOOT_CURR_SERIAL_BAUD */
    uint4   u4BootCurrentSerialBaudRate;
/*! When \ref bUseInitialProtocol is set, instructs the hardware bootstrap application to startup using this CAN bit rate. Select with \ref USE_BOOT_CURR_CAN_BITRATE */
    uint4   u4BootCANBitRate;
} S_RebootControlAttributes;

#pragma pack()

/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S RebootFramework(S_RebootControlAttributes const*);

#endif /* __REBOOTCONTROL_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
